// TimeTester - Stephen Stair 2007 - sgstair@akkit.org
// TimeTester is a simple test framework for measuring speed, comparing results, and testing for routine correctness on the DS.
// template.cpp: the main source file for the DS project

/* [MIT License segment]
Copyright (c) 2007 - Stephen Stair (sgstair@akkit.org)

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#include "nds.h"
#include <nds/arm9/console.h> //basic print funcionality
#include <stdio.h>
#include "template.h"
#include "TestTree.h"


extern u8 fontdata[];
u16 * subscreenptr =   ((u16 *) 0x06200000);

int font_ht, font_pitch, font_vpitch;
u8 * font_widtab;
u16 * font_lutab;
u16 * font_char16;


//////////////////////////////////////////////////////////////////////////
// General drawing functions

void pset( int x, int y, u16 color ) {
    if(x<0 || y<0 || x>255 || y>191) return;
    subscreenptr[x|(y<<8)]=color;
}

void ClearLines( int y1, int y2, u16 color ) {
    if(y1<0) y1=0;
    if(y2>191) y2=191;
    color |= 0x8000;
    u16 * pStart = subscreenptr + (y1<<8);
    int nCount = (y2-y1+1)<<8;
    while(nCount>0) {
        *(pStart++) = color;
        nCount--;
    }
}

void DrawFilledRect( int x1, int y1, int x2, int y2, u16 colorline, u16 colorfill )
{
    int x,y;
    int cx1,cy1,cx2,cy2;
    colorline |= 0x8000;
    colorfill |= 0x8000;
    if(y1>191 || y2<0) return;
    if(x1<0) cx1=0; else cx1=x1;
    if(x2>255) cx2=255; else cx2=x2;
    if(y1<0) cy1=0; else cy1=y1;
    if(y2>191) cy2=191; else cy2=y2;
    for(y=cy1+1;y<cy2;y++) {
        for(x=cx1+1;x<cx2;x++) subscreenptr[x|(y<<8)]=colorfill;
    }
    if(y1==cy1) for(x=cx1+1;x<cx2;x++) subscreenptr[x|(y1<<8)]=colorline;
    if(y2==cy2) for(x=cx1+1;x<cx2;x++) subscreenptr[x|(y2<<8)]=colorline;
    if(x1==cx1) for(y=cy1+1;y<cy2;y++) subscreenptr[x1|(y<<8)]=colorline;
    if(x2==cx2) for(y=cy1+1;y<cy2;y++) subscreenptr[x2|(y<<8)]=colorline;
}


//////////////////////////////////////////////////////////////////////////
// Font related functions

void SelectFont(int fontnum) {
    int numfonts = fontdata[0];
    int offset;
    if(fontnum<0 || fontnum>=numfonts) fontnum=0;
    offset = (int) (((u32 *)fontdata)[1+fontnum]);
    font_widtab = fontdata+offset+4;
    font_lutab = ((u16 *)(fontdata+offset+256+4));
    font_char16 = ((u16 *)(fontdata+offset+256+512+4));
    font_ht = fontdata[offset];
    font_pitch=fontdata[offset+1];
    font_vpitch=fontdata[offset+2];
}

void DrawCharacter(int x, int y, int charbase, int width, u16 color) {
    int i,j;
    for(i=0;i<width;i++) {
        for(j=0;j<font_ht;j++) {
            if(font_char16[charbase]&(1<<j)) pset(x,y+j,color);
        }
        x++;
        charbase++;
    }
}


int GetWidth(const char * string) {
    int wid = 0;
    while(*string) {
        wid+=font_widtab[*string]+font_pitch;
        string++;
    }
    return wid;
}

int GetFontHeight() {
    return font_vpitch;
}

void DrawString(int x, int y, const char * string, u16 color) {
    int ox=x;
    int wid;
    if(y<-font_vpitch || y>191) return;
    color |= 0x8000;
    while(*string) {
        if(y>191) break;
        wid=font_widtab[*string]+font_pitch;
        if(x+wid>256)  { x=ox; y+=font_vpitch; }
        DrawCharacter(x,y,font_lutab[*string],font_widtab[*string],color);
        x+=wid;
        string++;
    }
}

//////////////////////////////////////////////////////////////////////////
// timing related functions

void StartTimer()
{
    TIMER0_CR=0;
    TIMER1_CR=0;
	TIMER2_CR=0;
    TIMER0_DATA=0;
    TIMER1_DATA=0;
	TIMER2_DATA=0;
	TIMER1_CR=TIMER_CASCADE|TIMER_ENABLE;
	TIMER2_CR=TIMER_CASCADE|TIMER_ENABLE;
    TIMER0_CR=TIMER_ENABLE;
}

u64 StopTimer()  // returns number of clock cycles elapsed since StartTimer
{
    TIMER0_CR=0;
    TIMER1_CR=0;
	TIMER2_CR=0;
    return ((u64)TIMER0_DATA | ((u64)TIMER1_DATA<<16) | ((u64)TIMER2_DATA<<32))*2; // *2 because timer frequency is 1/2 cpu frequency
}

// Misc functions...
void DrawStatusBar(char * text)
{
	int renderline = 192 - 2 - GetFontHeight();
	ClearLines(renderline-2,192,0);
	DrawString(2,renderline,text,RGB15(31,31,31));
}




void VblHandler() {

}

bool COMPARE_TO_EXAMPLE = false;

//---------------------------------------------------------------------------------
int main(void) {
//---------------------------------------------------------------------------------

    SUB_BG2_X0=0;
    SUB_BG2_Y0=0;
    SUB_BG2_XDX=256;
    SUB_BG2_YDY=256;
    SUB_BG2_XDY=0;
    SUB_BG2_YDX=0;

    SelectFont(0);

    videoSetMode(0);	//not using the main screen
    vramSetMainBanks(VRAM_A_LCD,VRAM_B_LCD,VRAM_C_SUB_BG,VRAM_D_LCD);
	videoSetModeSub(MODE_5_2D | DISPLAY_BG2_ACTIVE);	


	SUB_BG2_CR = 0x4084;

    irqInit();
    irqSet(IRQ_VBLANK,VblHandler);
    irqEnable(IRQ_VBLANK);


    TestTree::RenderTree();

	*((vu16*)0x04000204) &= 0x7FFF; // give arm9 priority

	while(1) {
        swiWaitForVBlank();
  //      ClearLines(0,5,RGB5(31,0,31));

        scanKeys();

        if(keysDown()&KEY_A) {
			COMPARE_TO_EXAMPLE=true;
            irqDisable(IRQ_VBLANK);
            TestTree::RunTest();
            irqEnable(IRQ_VBLANK);
        }
		if(keysDown()&KEY_START) {
			COMPARE_TO_EXAMPLE=false;
			irqDisable(IRQ_VBLANK);
			TestTree::RunTest();
			irqEnable(IRQ_VBLANK);
		}        
		if(keysDown()&(KEY_B|KEY_LEFT)) {
            TestTree::NavLeft();
        }
        if(keysDown()&KEY_RIGHT) {
            TestTree::NavRight();
        }
        if(keysDown()&KEY_UP) {
            TestTree::NavUp();
        }
        if(keysDown()&KEY_DOWN) {
            TestTree::NavDown();
        }
	}

	return 0;
}
