// TimeTester - Stephen Stair 2007 - sgstair@akkit.org
// TimeTester is a simple test framework for measuring speed, comparing results, and testing for routine correctness on the DS.
// TestTree.cpp - the heart of the TimeTester, manages rendering of the tree structure, as well as movement within the tree and facilitates running tests.

/* [MIT License segment]
Copyright (c) 2007 - Stephen Stair (sgstair@akkit.org)

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#include <stdlib.h>
#include <string.h>
#include <algorithm>
#include "TestTree.h"
#include "template.h"

#define SCREEN_WIDTH    256
#define SCREEN_HEIGHT   192
#define SCREEN_X2       (SCREEN_WIDTH-1)
#define SEL_WIDTH       20

#define BLOCK_PADDING   3
#define BLOCK_SPACING   2

#define COLOR_NEUTRAL   RGB5(24,24,24)
#define COLOR_PASSED    RGB5(8,31,16)
#define COLOR_FAILED    RGB5(31,8,16)
#define COLOR_SELECTED  RGB5(16,24,31)

#define LINE_COLOR      RGB5(8,8,16)
#define TEXT_COLOR      RGB5(0,0,0)

char TreeNodeTempStr[256];


bool TestTreeNode::AmSelectedChild()
{
    if(parent) {
        if(parent->selected_index<(int)parent->subnodes.size()) {
            if(parent->subnodes[parent->selected_index]==this) {
                return parent->AmSelectedChild();
            } else {
                return false;
            }
        } else {
            return false;
        }
    } else {
        return true;
    }
}

TestTreeNode::TestTreeNode()
{
    panelcolor=COLOR_NEUTRAL;
    isExpanded=true;
    selected_index=0;
    startup_refcount=0;
    num_passed=num_tested=num_total=0;
    testname="root";
    testtext = "All Tests";
    parent=NULL;
}

int TestTreeNode::GetRenderHeight()
{
    if(!isExpanded) return BLOCK_PADDING*2 + GetFontHeight()*2;
    int height=-BLOCK_SPACING;
    size_t i;
    for(i=0;i<subnodes.size();i++) height+=subnodes[i]->GetRenderHeight() + BLOCK_SPACING;
    for(i=0;i<tests.size();i++) height += BLOCK_PADDING*2 + BLOCK_SPACING + GetFontHeight()*2;
    if(height<0) height=0;
    height += BLOCK_PADDING*2 + GetFontHeight();
    return height;
}
int TestTreeNode::GetSelectedCenterPos()
{
    if(!isExpanded) return BLOCK_PADDING + GetFontHeight();
    int pos=BLOCK_PADDING+GetFontHeight();
    size_t i;
    int n = 0;
    for(i=0;i<subnodes.size();i++) {
        if(n==selected_index) return pos+subnodes[i]->GetSelectedCenterPos();
        pos+=subnodes[i]->GetRenderHeight();
        pos+=BLOCK_SPACING;
        n++;
    }
    for(i=0;i<tests.size();i++) {
        if(n==selected_index) return pos+BLOCK_PADDING+GetFontHeight();
        pos += BLOCK_PADDING*2+GetFontHeight()*2+BLOCK_SPACING;
        n++;
    }
    return pos; // should never happen.
}
int TestTreeNode::Render(int xpos, int ypos)
{
    int myheight=GetRenderHeight();
    if(ypos>191) return myheight;
    u16 test_color, render_color;
    render_color=panelcolor;
    if(parent && !isExpanded) {
        if(AmSelectedChild()) {
            render_color=COLOR_SELECTED;
        }
    }
    DrawFilledRect(xpos,ypos,SCREEN_X2-xpos,ypos+myheight-1,LINE_COLOR,panelcolor);
    DrawFilledRect(xpos+1,ypos+1,xpos+SEL_WIDTH,ypos+myheight-2,render_color,render_color);
    xpos+=BLOCK_PADDING;
    ypos+=BLOCK_PADDING;
    DrawString(xpos,ypos,testname.c_str(),TEXT_COLOR);
    sprintf(TreeNodeTempStr,"%i/%i/%i",num_passed,num_tested,num_total);
    DrawString(SCREEN_X2-xpos-GetWidth(TreeNodeTempStr),ypos,TreeNodeTempStr,TEXT_COLOR);

    ypos += GetFontHeight();
    if(!isExpanded) {
        sprintf(TreeNodeTempStr,"%i Group%s, %i Test%s",subnodes.size(), subnodes.size()==1?"":"s", tests.size(), tests.size()==1?"":"s");
        DrawString(xpos,ypos,TreeNodeTempStr,TEXT_COLOR);
        return myheight;
    }
    size_t i;
    for(i=0;i<subnodes.size();i++) ypos += subnodes[i]->Render(xpos,ypos) + BLOCK_SPACING;
    for(i=0;i<tests.size();i++) {
        test_color=COLOR_NEUTRAL;
        if(test_run[i]) {
            if(test_passed[i]) test_color=COLOR_PASSED; else test_color=COLOR_FAILED;
        }
        render_color=test_color;
        if(((selected_index-subnodes.size()) == i) && AmSelectedChild()) render_color=COLOR_SELECTED;
        DrawFilledRect(xpos,ypos,SCREEN_X2-xpos,ypos+BLOCK_PADDING*2+GetFontHeight()*2,LINE_COLOR,test_color);
        DrawFilledRect(xpos+1,ypos+1,xpos+SEL_WIDTH,ypos+BLOCK_PADDING*2+GetFontHeight()*2-1,render_color,render_color);
        ypos+=BLOCK_PADDING;
        DrawString(xpos+BLOCK_PADDING,ypos,test_name[i].c_str(),TEXT_COLOR);
        ypos+=GetFontHeight();
        DrawString(xpos+BLOCK_PADDING,ypos,test_result[i].c_str(),TEXT_COLOR);
        ypos+=GetFontHeight()+BLOCK_PADDING+BLOCK_SPACING;
    }
    return myheight;
}
int TestTreeNode::GetNumElements()
{
    return (int) (subnodes.size() + tests.size());
}
void TestTreeNode::RunTests()
{
    int i;
    int nSubnodes, nTests;
    nSubnodes=(int)subnodes.size();
    nTests=(int)tests.size();
    for(i=0;i<nSubnodes;i++) subnodes[i]->RunTests();
    startup_refcount++;
    if(startup_refcount==1 && startupfunc) startupfunc();
    for(i=0;i<nTests;i++) RunTest(i+nSubnodes);
    startup_refcount--;
    if(startup_refcount==0 && shutdownfunc) shutdownfunc(); 
}
void TestTreeNode::RunSelectedTest()
{
    if(isExpanded) 
        if(selected_index<(int)subnodes.size()) {
            subnodes[selected_index]->RunSelectedTest();
        } else {
            RunTest(selected_index);
        }
    else {
        RunTests();
    }
        
}
void TestTreeNode::RunTest(int test)
{
    if(test<(int)subnodes.size()) {
        subnodes[test]->RunTests();
    } else {
        test-=(int)subnodes.size();
        if(test<(int)tests.size()) {
            startup_refcount++;
            if(startup_refcount==1 && startupfunc) startupfunc();
            {
                bool success;
                test_result[test] = tests[test]->RunTest(success);
                test_run[test] = true;
                test_passed[test] = success;

            }
            startup_refcount--;
            if(startup_refcount==0 && shutdownfunc) shutdownfunc(); 
            EvalDownColor();
            TestTree::RenderTree();
        }
    }
}

void TestTreeNode::EvalMyColor()
{
    int r,g,b,count;
    r=g=b=count=0;
    u16 color;
    size_t i;
    num_tested=0;
    num_passed=0;
    num_total=0;

    for(i=0;i<subnodes.size();i++) {
        num_tested+=subnodes[i]->num_tested;
        num_passed+=subnodes[i]->num_passed;
        num_total+=subnodes[i]->num_total;
        color=subnodes[i]->panelcolor;
        r+=color&31; g+=(color>>5)&31; b+=(color>>10)&31; count++;
    }
    for(i=0;i<tests.size();i++) {
        color=COLOR_NEUTRAL;
        num_total++;
        if(test_run[i]) {
            num_tested++;
            if(test_passed[i]) { color=COLOR_PASSED; num_passed++; } else color=COLOR_FAILED;
        }
        r+=color&31; g+=(color>>5)&31; b+=(color>>10)&31; count++;
    }
    if(count==0) {
        panelcolor=COLOR_NEUTRAL;
    } else {
        r = (r+count-1)/count;
        g = (g+count-1)/count;
        b = (b+count-1)/count;
        panelcolor = r | (g<<5) | (b<<10);
    }
}
void TestTreeNode::EvalDownColor()
{
    EvalMyColor();
    if(parent) parent->EvalDownColor();
}
void TestTreeNode::EvalUpColor()
{
    size_t i;
    for(i=0;i<subnodes.size();i++) {
        subnodes[i]->EvalUpColor();
    }
    EvalMyColor();
}
/*
std::string testname;
std::string testtext;

u16 panelcolor;
bool isExpanded;
int selected_index;

void (*startupfunc)();
void (*shutdownfunc)();
std::vector<TestTreeNode *> subnodes;
std::vector<TestNode *> tests;
std::vector<bool> test_passed;
std::vector<bool> test_run;
std::vector<std::string> test_name;
std::vector<std::string> test_result;
};*/



TestTreeNode TestTree::root;

void TestTree::RegisterTest(TestNode * test, std::string testname, std::string parent)
{
    TestTreeNode * node = GetExistingCategory(parent);
    if(!node) node = GetNewCategory(parent,"root");
    node->tests.push_back(test);
    node->test_passed.push_back(false);
    node->test_run.push_back(false);
    node->test_name.push_back(testname);
    node->test_result.push_back("(Not Run)");
    node->EvalDownColor();
}

void TestTree::RegisterCategory(std::string categoryname, std::string categorytext, std::string parent, void (*startup)(), void (*shutdown)() )
{
    TestTreeNode * node = GetNewCategory(categoryname,parent);
    node->testtext=categorytext;
    node->startupfunc=startup;
    node->shutdownfunc=shutdown;
}

void TestTree::RenderTree()
{
    int centery = root.GetSelectedCenterPos();
    int starty=96-centery;
    ClearLines(0,starty,RGB5(0,0,8));
    ClearLines(starty+root.GetRenderHeight(),191,RGB5(0,0,8));
    root.Render(0,starty);
}
void TestTree::RunTest()
{
    root.RunSelectedTest();
}
void TestTree::NavDown()
{
    TestTreeNode * node = &root;
    int nSubnodes, nSelnode, iterations=0, wasExpanded;
    while(iterations++<100) { // first find "current" node
        nSelnode=node->selected_index;
        nSubnodes=(int)node->subnodes.size();
        if(nSelnode<nSubnodes) {
            if(node->subnodes[nSelnode]->isExpanded) {
                node = node->subnodes[nSelnode];
            } else break;
        } else break;
    }
    // have current node, now move cursor down.
    wasExpanded=0;
    iterations=0;
    while(iterations++<100) { // move down the tree towards root if we are in the last position, otherwise just move down one item
        nSelnode=node->selected_index;
        nSubnodes=(int)node->subnodes.size();
        if((nSelnode+1)<node->GetNumElements()) {
            node->selected_index=++nSelnode;
            if(nSelnode<nSubnodes) {
                if(node->subnodes[nSelnode]->isExpanded) wasExpanded=true;
            }
            break;
        } else {
            node=node->parent;
            if(!node) return;
        }
    }
    if(wasExpanded) { // landed on an expanded tree, so move up in the tree until we get to the next visible item.
        iterations=0;
        while(iterations++<100) {
            nSelnode=node->selected_index;
            nSubnodes=(int)node->subnodes.size();
            if(nSelnode<nSubnodes) {
                if(node->subnodes[nSelnode]->isExpanded) {
                    node=node->subnodes[nSelnode];
                    node->selected_index=0;
                } else {
                    break;
                }
            } else {
                break;
            }
        }
    }
    RenderTree();
}
void TestTree::NavUp()
{
    TestTreeNode * node = &root;
    int nSubnodes, nSelnode, iterations=0, wasExpanded;
    while(iterations++<100) { // first find "current" node
        nSelnode=node->selected_index;
        nSubnodes=(int)node->subnodes.size();
        if(nSelnode<nSubnodes) {
            if(node->subnodes[nSelnode]->isExpanded) {
                node = node->subnodes[nSelnode];
            } else break;
        } else break;
    }
    // have current node, now move cursor up.
    wasExpanded=0;
    iterations=0;
    while(iterations++<100) { // move down the tree towards root if we are in the 0'th position, otherwise just move up one item
        nSelnode=node->selected_index;
        nSubnodes=(int)node->subnodes.size();
        if(nSelnode>0) {
            node->selected_index=--nSelnode;
            if(nSelnode<nSubnodes) {
                if(node->subnodes[nSelnode]->isExpanded) wasExpanded=true;
            }
            break;
        } else {
            node=node->parent;
            if(!node) return;
        }
    }
    if(wasExpanded) { // landed on an expanded tree, so move up in the tree until we get to the previous visible item.
        iterations=0;
        while(iterations++<100) {
            nSelnode=node->selected_index;
            nSubnodes=(int)node->subnodes.size();
            if(nSelnode<nSubnodes) {
                if(node->subnodes[nSelnode]->isExpanded) {
                    node=node->subnodes[nSelnode];
                    node->selected_index=node->GetNumElements()-1;
                } else {
                    break;
                }
            } else {
                break;
            }
        }
    }
    RenderTree();
}
void TestTree::NavLeft()
{
    TestTreeNode * node = &root;
    int nSubnodes, nSelnode, iterations=0;
    while(iterations++<100) {
        nSelnode=node->selected_index;
        nSubnodes=(int)node->subnodes.size();
        if(nSelnode<nSubnodes) {
            if(node->subnodes[nSelnode]->isExpanded) {
                node=node->subnodes[nSelnode];
            } else {
                if(node!=&root) node->isExpanded=false;
                RenderTree();
                return;
            }
        } else  {
            if(node!=&root) node->isExpanded=false;
            RenderTree();
            return;            
        }
    }

}
void TestTree::NavRight()
{
    TestTreeNode * node = &root;
    int nSubnodes, nSelnode, iterations=0;
    while(iterations++<100) {
        nSelnode=node->selected_index;
        nSubnodes=(int)node->subnodes.size();
        if(nSelnode<nSubnodes) {
            if(node->subnodes[nSelnode]->isExpanded) {
                node=node->subnodes[nSelnode];
            } else {
                node=node->subnodes[nSelnode];
                if(node->GetNumElements()>0) {
                    node->isExpanded=true;
                    node->selected_index=0;
                    RenderTree();
                }
                return;
            }
        } else {
            return;
        }
    }
}

TestTreeNode * TestTree::GetExistingCategory(std::string name, TestTreeNode * base)
{
    if(!base) base=&root;
    if(base->testname==name) return base;
    size_t i;
    TestTreeNode * result;
    for(i=0;i<base->subnodes.size();i++) {
        result = GetExistingCategory(name,base->subnodes[i]);
        if(result) return result;
    }
    return NULL;
}
TestTreeNode * TestTree::GetNewCategory(std::string name, std::string parent)
{
    TestTreeNode * node, * pnode;
    if((node=GetExistingCategory(name))) {
        if(node->parent) {
            if(node->parent->testname!=parent) { // relocate node!
                pnode = GetExistingCategory(parent);
                if(pnode) {// found suitably named parent!
                    node->parent->subnodes.erase(std::find(node->parent->subnodes.begin(), node->parent->subnodes.end(), node));
                    node->parent=pnode;
                    pnode->subnodes.push_back(node);
                    root.EvalUpColor();
                }
            }
        }
        return node;
    }
    pnode = GetExistingCategory(parent);
    if(!pnode) pnode=&root;
    node = new TestTreeNode();
    pnode->subnodes.push_back(node);
    node->parent=pnode;
    node->isExpanded=false;
    node->testname=name;
    node->testtext=name;
    node->EvalDownColor();
    return node;
}


